<?php
/**
 * ============================================
 * SHADOW PRO - ADMIN DASHBOARD (admin.php)
 * ============================================
 * 
 * FEATURES:
 * - Premium Neon/Shadow 3D UI
 * - Secure login with CSRF protection
 * - Optional IP restriction for admin access
 * - Complete dashboard for managing redirects, whitelist, analytics
 * - Real-time statistics and monitoring
 */

// ============================================
// SECURITY: DEFINE SYSTEM CONSTANT
// ============================================
define('SHADOW_PRO', true);

// ============================================
// INCLUDE CONFIGURATION
// ============================================
require_once 'config.php';

// ============================================
// DATABASE CONNECTION
// ============================================
$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_error) {
    die("Database connection failed: " . $db->connect_error);
}

// ============================================
// AUTHENTICATION CLASS
// ============================================
class ShadowAuth {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    public function login($username, $password) {
        // Check IP restriction if enabled
        $restrict = $this->db->query("SELECT setting_value FROM shadow_settings WHERE setting_key='admin_ip_restrict'")->fetch_assoc()['setting_value'];
        
        if ($restrict == '1') {
            $allowed_ips = $this->db->query("SELECT setting_value FROM shadow_settings WHERE setting_key='admin_allowed_ips'")->fetch_assoc()['setting_value'];
            $client_ip = $_SERVER['REMOTE_ADDR'];
            
            if (!in_array($client_ip, explode(',', $allowed_ips))) {
                return ['success' => false, 'error' => 'IP address not authorized'];
            }
        }
        
        // Get user from database
        $stmt = $this->db->prepare("SELECT * FROM shadow_admins WHERE username = ? AND is_active = 1");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $user = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if ($user && password_verify($password, $user['password_hash'])) {
            // Update login info
            $update = $this->db->prepare("UPDATE shadow_admins SET last_login = NOW(), login_ip = ? WHERE id = ?");
            $update->bind_param("si", $_SERVER['REMOTE_ADDR'], $user['id']);
            $update->execute();
            $update->close();
            
            // Set secure session
            $_SESSION['admin_id'] = $user['id'];
            $_SESSION['admin_user'] = $user['username'];
            $_SESSION['admin_token'] = bin2hex(random_bytes(32));
            $_SESSION['login_time'] = time();
            
            // Regenerate session ID
            session_regenerate_id(true);
            
            return ['success' => true];
        }
        
        return ['success' => false, 'error' => 'Invalid username or password'];
    }
    
    public function isLoggedIn() {
        if (!isset($_SESSION['admin_id'], $_SESSION['admin_token'])) {
            return false;
        }
        
        // Check session timeout
        if (time() - $_SESSION['login_time'] > SESSION_TIMEOUT) {
            $this->logout();
            return false;
        }
        
        // Update last activity
        $_SESSION['login_time'] = time();
        return true;
    }
    
    public function logout() {
        session_unset();
        session_destroy();
        session_write_close();
        setcookie(session_name(), '', time() - 3600, '/');
    }
    
    public function generateCSRF() {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }
    
    public function validateCSRF($token) {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}

// ============================================
// INITIALIZE AUTHENTICATION
// ============================================
$auth = new ShadowAuth($db);

// Handle logout
if (isset($_GET['action']) && $_GET['action'] == 'logout') {
    $auth->logout();
    header('Location: admin.php');
    exit;
}

// Handle login form submission
$login_error = '';
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['login'])) {
    $result = $auth->login($_POST['username'], $_POST['password']);
    if ($result['success']) {
        header('Location: admin.php');
        exit;
    } else {
        $login_error = $result['error'];
    }
}

// ============================================
// CHECK AUTHENTICATION - SHOW LOGIN IF NOT LOGGED IN
// ============================================
if (!$auth->isLoggedIn()) {
    // Display login page with Neon UI
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Shadow PRO - Admin Login</title>
        <style>
            /* NEON/SHADOW 3D LOGIN STYLES */
            :root {
                --neon-blue: #0ff0fc;
                --neon-purple: #bc13fe;
                --neon-green: #39ff14;
                --neon-red: #ff073a;
                --dark-bg: #0a0a0f;
                --darker-bg: #050508;
                --card-bg: rgba(15, 15, 25, 0.9);
            }
            
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
                font-family: 'Segoe UI', 'Arial', sans-serif;
            }
            
            body {
                background: var(--dark-bg);
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
                overflow: hidden;
                position: relative;
            }
            
            .login-background {
                position: absolute;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: 
                    radial-gradient(circle at 20% 30%, var(--neon-purple) 0%, transparent 50%),
                    radial-gradient(circle at 80% 70%, var(--neon-blue) 0%, transparent 50%);
                filter: blur(100px);
                opacity: 0.3;
                animation: pulse 15s infinite alternate;
            }
            
            @keyframes pulse {
                0%, 100% { transform: rotate(0deg); }
                50% { transform: rotate(180deg); }
            }
            
            .login-container {
                background: var(--card-bg);
                backdrop-filter: blur(10px);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 20px;
                padding: 40px;
                width: 100%;
                max-width: 400px;
                box-shadow: 
                    0 0 30px rgba(11, 240, 252, 0.3),
                    0 0 60px rgba(188, 19, 254, 0.2),
                    inset 0 1px 0 rgba(255, 255, 255, 0.1);
                position: relative;
                z-index: 1;
                animation: float 6s ease-in-out infinite;
            }
            
            @keyframes float {
                0%, 100% { transform: translateY(0) rotateX(0); }
                50% { transform: translateY(-20px) rotateX(5deg); }
            }
            
            .login-header {
                text-align: center;
                margin-bottom: 40px;
            }
            
            .login-header h1 {
                color: white;
                font-size: 2.5rem;
                margin-bottom: 10px;
                text-shadow: 
                    0 0 10px var(--neon-blue),
                    0 0 20px var(--neon-blue),
                    0 0 30px var(--neon-blue);
                letter-spacing: 3px;
            }
            
            .login-header .subtitle {
                color: rgba(255, 255, 255, 0.7);
                font-size: 0.9rem;
                letter-spacing: 2px;
                text-transform: uppercase;
            }
            
            .form-group {
                margin-bottom: 25px;
                position: relative;
            }
            
            .form-group label {
                display: block;
                color: rgba(255, 255, 255, 0.6);
                margin-bottom: 8px;
                font-size: 0.9rem;
                letter-spacing: 1px;
            }
            
            .form-group input {
                width: 100%;
                padding: 15px 20px;
                background: rgba(255, 255, 255, 0.05);
                border: 2px solid rgba(255, 255, 255, 0.1);
                border-radius: 10px;
                color: white;
                font-size: 1rem;
                transition: all 0.3s;
            }
            
            .form-group input:focus {
                outline: none;
                border-color: var(--neon-blue);
                box-shadow: 0 0 20px rgba(11, 240, 252, 0.3);
            }
            
            .login-btn {
                width: 100%;
                padding: 16px;
                background: linear-gradient(45deg, var(--neon-purple), var(--neon-blue));
                border: none;
                border-radius: 10px;
                color: white;
                font-size: 1.1rem;
                font-weight: bold;
                letter-spacing: 2px;
                cursor: pointer;
                transition: all 0.3s;
                text-transform: uppercase;
                position: relative;
                overflow: hidden;
            }
            
            .login-btn:hover {
                transform: translateY(-2px);
                box-shadow: 
                    0 10px 20px rgba(11, 240, 252, 0.4),
                    0 0 30px rgba(188, 19, 254, 0.3);
            }
            
            .error-message {
                background: rgba(255, 50, 50, 0.2);
                border: 1px solid rgba(255, 50, 50, 0.5);
                border-radius: 10px;
                padding: 15px;
                margin-bottom: 20px;
                color: #ff6b6b;
                text-align: center;
                animation: errorShake 0.5s;
            }
            
            @keyframes errorShake {
                0%, 100% { transform: translateX(0); }
                25% { transform: translateX(-10px); }
                75% { transform: translateX(10px); }
            }
        </style>
    </head>
    <body>
        <div class="login-background"></div>
        <div class="login-container">
            <div class="login-header">
                <h1>SHADOW PRO</h1>
                <div class="subtitle">Admin Access Panel</div>
            </div>
            
            <?php if ($login_error): ?>
            <div class="error-message">
                <?php echo htmlspecialchars($login_error); ?>
            </div>
            <?php endif; ?>
            
            <form method="POST" action="">
                <input type="hidden" name="login" value="1">
                
                <div class="form-group">
                    <label>USERNAME</label>
                    <input type="text" name="username" required autofocus>
                </div>
                
                <div class="form-group">
                    <label>PASSWORD</label>
                    <input type="password" name="password" required>
                </div>
                
                <button type="submit" class="login-btn">
                    ACCESS DASHBOARD
                </button>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// ============================================
// ADMIN DASHBOARD (LOGGED IN)
// ============================================

// Handle POST actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $csrf_token = $_POST['csrf_token'] ?? '';
    if (!$auth->validateCSRF($csrf_token)) {
        die('CSRF validation failed');
    }
    
    // Handle different actions
    if (isset($_POST['update_settings'])) {
        foreach ($_POST['settings'] as $key => $value) {
            $stmt = $db->prepare("UPDATE shadow_settings SET setting_value = ? WHERE setting_key = ?");
            $stmt->bind_param("ss", $value, $key);
            $stmt->execute();
            $stmt->close();
        }
        $_SESSION['success_message'] = "Settings updated successfully!";
    }
    
    if (isset($_POST['add_link'])) {
        $stmt = $db->prepare("INSERT INTO shadow_links (url, title, priority, weight, status) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("ssiis", 
            $_POST['url'],
            $_POST['title'],
            $_POST['priority'],
            $_POST['weight'],
            $_POST['status']
        );
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "Redirect link added!";
    }
    
    if (isset($_POST['add_whitelist'])) {
        $stmt = $db->prepare("INSERT IGNORE INTO shadow_whitelist (ip_address, note, added_by) VALUES (?, ?, ?)");
        $stmt->bind_param("sss", 
            $_POST['ip_address'],
            $_POST['note'],
            $_SESSION['admin_user']
        );
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "IP added to whitelist!";
    }
    
    if (isset($_POST['delete_link'])) {
        $stmt = $db->prepare("DELETE FROM shadow_links WHERE id = ?");
        $stmt->bind_param("i", $_POST['link_id']);
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "Link deleted!";
    }
    
    if (isset($_POST['delete_whitelist'])) {
        $stmt = $db->prepare("DELETE FROM shadow_whitelist WHERE id = ?");
        $stmt->bind_param("i", $_POST['whitelist_id']);
        $stmt->execute();
        $stmt->close();
        $_SESSION['success_message'] = "IP removed from whitelist!";
    }
    
    // Redirect to prevent form resubmission
    header('Location: admin.php');
    exit;
}

// ============================================
// FETCH DATA FOR DASHBOARD
// ============================================

// Get settings
$settings = [];
$settings_result = $db->query("SELECT setting_key, setting_value FROM shadow_settings");
while ($row = $settings_result->fetch_assoc()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Get redirect links
$links = $db->query("SELECT * FROM shadow_links ORDER BY status, priority");

// Get whitelist
$whitelist = $db->query("SELECT * FROM shadow_whitelist ORDER BY added_at DESC");

// Get statistics
$stats = [];

// Total visits
$result = $db->query("SELECT COUNT(*) as total FROM shadow_visits");
$stats['total_visits'] = $result->fetch_assoc()['total'];

// Today's visits
$result = $db->query("SELECT COUNT(*) as today FROM shadow_visits WHERE DATE(created_at) = CURDATE()");
$stats['today_visits'] = $result->fetch_assoc()['today'];

// Classification breakdown (last 24h)
$result = $db->query("SELECT classification, COUNT(*) as count FROM shadow_visits WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) GROUP BY classification");
$classification_stats = [];
while ($row = $result->fetch_assoc()) {
    $classification_stats[$row['classification']] = $row['count'];
}

// Link performance
$result = $db->query("SELECT l.id, l.title, l.status, COUNT(v.id) as redirects, l.successful_redirects FROM shadow_links l LEFT JOIN shadow_visits v ON l.id = v.redirect_link_id GROUP BY l.id ORDER BY redirects DESC");
$link_stats = $result->fetch_all(MYSQLI_ASSOC);

// Country stats
$result = $db->query("SELECT country_code, COUNT(*) as count FROM shadow_visits WHERE country_code IS NOT NULL GROUP BY country_code ORDER BY count DESC LIMIT 10");
$country_stats = $result->fetch_all(MYSQLI_ASSOC);

// Recent visits
$result = $db->query("SELECT v.*, l.title as link_title FROM shadow_visits v LEFT JOIN shadow_links l ON v.redirect_link_id = l.id ORDER BY v.created_at DESC LIMIT 10");
$recent_visits = $result->fetch_all(MYSQLI_ASSOC);

$csrf_token = $auth->generateCSRF();
$success_message = $_SESSION['success_message'] ?? '';
unset($_SESSION['success_message']);

?>
<!DOCTYPE html>
<html lang="en" class="shadow-dashboard">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shadow PRO Dashboard</title>
    <link rel="stylesheet" href="assets/css/shadow-ui.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* NEON/SHADOW 3D DASHBOARD STYLES */
        :root {
            --neon-blue: #0ff0fc;
            --neon-purple: #bc13fe;
            --neon-green: #39ff14;
            --neon-red: #ff073a;
            --dark-bg: #0a0a0f;
            --darker-bg: #050508;
            --card-bg: rgba(15, 15, 25, 0.85);
            --card-border: rgba(255, 255, 255, 0.1);
            --text-primary: #ffffff;
            --text-secondary: #a0a0c0;
            --shadow-color: rgba(11, 240, 252, 0.2);
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background: var(--dark-bg);
            color: var(--text-primary);
            font-family: 'Segoe UI', 'Arial', sans-serif;
            min-height: 100vh;
            overflow-x: hidden;
        }
        
        .dashboard-container {
            display: flex;
            flex-direction: column;
            min-height: 100vh;
        }
        
        /* Header */
        .dashboard-header {
            background: rgba(10, 10, 15, 0.9);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid rgba(11, 240, 252, 0.2);
            padding: 15px 30px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            position: sticky;
            top: 0;
            z-index: 1000;
            box-shadow: 0 5px 30px rgba(0, 0, 0, 0.5);
        }
        
        .header-left {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .logo {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 1.8rem;
            letter-spacing: 3px;
            text-shadow: 
                0 0 10px var(--neon-blue),
                0 0 20px var(--neon-blue),
                0 5px 5px rgba(0, 0, 0, 0.5);
        }
        
        .logo i {
            color: var(--neon-purple);
            animation: logoSpin 10s linear infinite;
        }
        
        @keyframes logoSpin {
            0% { transform: rotateY(0); }
            100% { transform: rotateY(360deg); }
        }
        
        .logo-pro {
            background: linear-gradient(45deg, var(--neon-purple), var(--neon-blue);
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
            font-weight: bold;
        }
        
        .version {
            background: rgba(11, 240, 252, 0.1);
            padding: 2px 10px;
            border-radius: 10px;
            font-size: 0.8rem;
            color: var(--neon-blue);
            border: 1px solid rgba(11, 240, 252, 0.3);
        }
        
        .nav-tabs {
            display: flex;
            gap: 5px;
            background: rgba(20, 20, 30, 0.8);
            border-radius: 15px;
            padding: 5px;
        }
        
        .nav-tab {
            background: transparent;
            border: none;
            color: var(--text-secondary);
            padding: 12px 20px;
            border-radius: 12px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s;
            font-size: 0.95rem;
            letter-spacing: 1px;
        }
        
        .nav-tab:hover {
            color: var(--text-primary);
            background: rgba(255, 255, 255, 0.05);
        }
        
        .nav-tab.active {
            background: linear-gradient(45deg, var(--neon-purple), var(--neon-blue));
            color: white;
            box-shadow: 0 0 15px rgba(11, 240, 252, 0.3);
        }
        
        .user-menu {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-name {
            display: flex;
            align-items: center;
            gap: 8px;
            color: var(--text-secondary);
        }
        
        .logout-btn {
            color: var(--neon-red);
            text-decoration: none;
            font-size: 1.2rem;
            transition: all 0.3s;
        }
        
        .logout-btn:hover {
            transform: scale(1.2);
            text-shadow: 0 0 10px var(--neon-red);
        }
        
        /* Main Content */
        .dashboard-content {
            flex: 1;
            padding: 30px;
            max-width: 1600px;
            margin: 0 auto;
            width: 100%;
        }
        
        .notification {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            animation: slideDown 0.5s;
            background: rgba(57, 255, 20, 0.1);
            border: 1px solid rgba(57, 255, 20, 0.3);
            color: var(--neon-green);
        }
        
        @keyframes slideDown {
            from { transform: translateY(-20px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }
        
        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: var(--card-bg);
            border: 1px solid var(--card-border);
            border-radius: 15px;
            padding: 20px;
            display: flex;
            align-items: center;
            gap: 20px;
            position: relative;
            overflow: hidden;
            transition: all 0.3s;
        }
        
        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 3px;
            background: linear-gradient(90deg, transparent, currentColor, transparent);
            opacity: 0.5;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px var(--shadow-color);
        }
        
        .stat-card.glow-blue { color: var(--neon-blue); }
        .stat-card.glow-green { color: var(--neon-green); }
        .stat-card.glow-purple { color: var(--neon-purple); }
        .stat-card.glow-red { color: var(--neon-red); }
        
        .stat-icon {
            font-size: 2.5rem;
            filter: drop-shadow(0 0 10px currentColor);
        }
        
        .stat-info {
            flex: 1;
        }
        
        .stat-info h3 {
            font-size: 0.9rem;
            color: var(--text-secondary);
            margin-bottom: 5px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .stat-number {
            font-size: 2rem;
            font-weight: bold;
            line-height: 1;
            margin-bottom: 5px;
        }
        
        .stat-percentage, .stat-trend {
            font-size: 0.9rem;
            opacity: 0.8;
        }
        
        /* Cards */
        .card {
            background: var(--card-bg);
            border: 1px solid var(--card-border);
            border-radius: 15px;
            overflow: hidden;
            margin-bottom: 20px;
            transition: all 0.3s;
        }
        
        .card.glow {
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.3);
        }
        
        .card.glow:hover {
            box-shadow: 
                0 10px 30px rgba(11, 240, 252, 0.2),
                0 0 50px rgba(188, 19, 254, 0.1);
        }
        
        .card-header {
            padding: 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .card-header h3 {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 1.2rem;
        }
        
        .card-body {
            padding: 20px;
        }
        
        /* Forms */
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .form-group label {
            color: var(--text-secondary);
            font-size: 0.9rem;
            font-weight: 500;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 8px;
            padding: 12px 15px;
            color: var(--text-primary);
            font-size: 1rem;
            transition: all 0.3s;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--neon-blue);
            box-shadow: 0 0 15px rgba(11, 240, 252, 0.2);
        }
        
        .checkbox {
            display: flex;
            align-items: center;
            gap: 10px;
            cursor: pointer;
        }
        
        .checkbox input[type="checkbox"] {
            width: 18px;
            height: 18px;
            accent-color: var(--neon-blue);
        }
        
        .help-text {
            font-size: 0.85rem;
            color: var(--text-secondary);
            opacity: 0.7;
        }
        
        /* Buttons */
        .btn-primary {
            background: linear-gradient(45deg, var(--neon-purple), var(--neon-blue));
            border: none;
            color: white;
            padding: 12px 25px;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s;
            letter-spacing: 1px;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 
                0 10px 20px rgba(11, 240, 252, 0.4),
                0 0 30px rgba(188, 19, 254, 0.3);
        }
        
        .btn-danger {
            background: linear-gradient(45deg, var(--neon-red), #ff416c);
        }
        
        .btn-icon {
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid rgba(255, 255, 255, 0.1);
            color: var(--text-secondary);
            width: 35px;
            height: 35px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-icon:hover {
            background: rgba(11, 240, 252, 0.1);
            color: var(--neon-blue);
            border-color: var(--neon-blue);
        }
        
        .btn-icon.danger:hover {
            background: rgba(255, 7, 58, 0.1);
            color: var(--neon-red);
            border-color: var(--neon-red);
        }
        
        /* Tables */
        .table-responsive {
            overflow-x: auto;
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table th {
            background: rgba(0, 0, 0, 0.3);
            padding: 15px;
            text-align: left;
            color: var(--text-secondary);
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.85rem;
            letter-spacing: 1px;
        }
        
        .data-table td {
            padding: 15px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.05);
        }
        
        .data-table tr:hover {
            background: rgba(255, 255, 255, 0.02);
        }
        
        .status-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
            letter-spacing: 0.5px;
        }
        
        .status-active {
            background: rgba(57, 255, 20, 0.1);
            color: var(--neon-green);
            border: 1px solid rgba(57, 255, 20, 0.3);
        }
        
        .status-backup {
            background: rgba(11, 240, 252, 0.1);
            color: var(--neon-blue);
            border: 1px solid rgba(11, 240, 252, 0.3);
        }
        
        .status-emergency {
            background: rgba(255, 7, 58, 0.1);
            color: var(--neon-red);
            border: 1px solid rgba(255, 7, 58, 0.3);
        }
        
        .status-disabled {
            background: rgba(255, 255, 255, 0.05);
            color: var(--text-secondary);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .classification-human { color: var(--neon-green); }
        .classification-bot { color: var(--neon-red); }
        .classification-suspicious { color: #ffaa00; }
        .classification-blocked { color: var(--neon-red); }
        
        /* Footer */
        .dashboard-footer {
            background: rgba(10, 10, 15, 0.9);
            border-top: 1px solid rgba(11, 240, 252, 0.2);
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.9rem;
            color: var(--text-secondary);
        }
        
        .status-indicator {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .status-indicator i {
            color: <?php echo $settings['system_enabled'] == '1' ? 'var(--neon-green)' : 'var(--neon-red)'; ?>;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .server-time {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        /* Tab Management */
        .tab-content {
            display: none;
            animation: fadeIn 0.5s;
        }
        
        .tab-content.active {
            display: block;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Traffic Distribution */
        .traffic-distribution {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .traffic-item {
            flex: 1;
            min-width: 200px;
        }
        
        .traffic-bar {
            height: 10px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 5px;
            overflow: hidden;
            margin-top: 10px;
        }
        
        .bar-fill {
            height: 100%;
            border-radius: 5px;
        }
        
        .bar-human { background: var(--neon-green); }
        .bar-bot { background: var(--neon-red); }
        .bar-suspicious { background: #ffaa00; }
        .bar-blocked { background: var(--neon-purple); }
        
        /* Country Grid */
        .countries-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
            gap: 15px;
        }
        
        .country-item {
            background: rgba(255, 255, 255, 0.03);
            border: 1px solid rgba(255, 255, 255, 0.05);
            border-radius: 10px;
            padding: 15px;
            text-align: center;
            transition: all 0.3s;
        }
        
        .country-item:hover {
            transform: translateY(-3px);
            border-color: var(--neon-blue);
            box-shadow: 0 5px 15px rgba(11, 240, 252, 0.2);
        }
        
        .country-flag {
            font-size: 2rem;
            margin-bottom: 10px;
            filter: drop-shadow(0 0 5px rgba(255, 255, 255, 0.5));
        }
        
        .country-name {
            font-size: 0.9rem;
            margin-bottom: 5px;
            color: var(--text-secondary);
        }
        
        .country-count {
            font-weight: bold;
            font-size: 1.2rem;
            color: var(--neon-blue);
        }
        
        /* Settings Sections */
        .settings-section {
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.05);
        }
        
        .settings-section h4 {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 20px;
            color: var(--neon-blue);
            font-size: 1.1rem;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .dashboard-header {
                flex-direction: column;
                gap: 15px;
                padding: 15px;
            }
            
            .nav-tabs {
                width: 100%;
                overflow-x: auto;
            }
            
            .dashboard-content {
                padding: 15px;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-container">
        <!-- Header -->
        <header class="dashboard-header">
            <div class="header-left">
                <h1 class="logo">
                    <i class="fas fa-ghost"></i>
                    <span class="logo-text">SHADOW</span>
                    <span class="logo-pro">PRO</span>
                </h1>
                <div class="version">v<?php echo SYSTEM_VERSION; ?></div>
            </div>
            
            <div class="header-center">
                <nav class="nav-tabs">
                    <button class="nav-tab active" data-tab="dashboard">
                        <i class="fas fa-tachometer-alt"></i> Dashboard
                    </button>
                    <button class="nav-tab" data-tab="links">
                        <i class="fas fa-link"></i> Redirect Links
                    </button>
                    <button class="nav-tab" data-tab="analytics">
                        <i class="fas fa-chart-bar"></i> Analytics
                    </button>
                    <button class="nav-tab" data-tab="whitelist">
                        <i class="fas fa-user-shield"></i> Whitelist
                    </button>
                    <button class="nav-tab" data-tab="settings">
                        <i class="fas fa-cogs"></i> Settings
                    </button>
                </nav>
            </div>
            
            <div class="header-right">
                <div class="user-menu">
                    <span class="user-name">
                        <i class="fas fa-user-circle"></i>
                        <?php echo htmlspecialchars($_SESSION['admin_user']); ?>
                    </span>
                    <a href="?action=logout" class="logout-btn" title="Logout">
                        <i class="fas fa-sign-out-alt"></i>
                    </a>
                </div>
            </div>
        </header>

        <!-- Main Content -->
        <main class="dashboard-content">
            <?php if ($success_message): ?>
            <div class="notification">
                <i class="fas fa-check-circle"></i>
                <?php echo htmlspecialchars($success_message); ?>
            </div>
            <?php endif; ?>

            <!-- Dashboard Tab -->
            <div class="tab-content active" id="dashboard-tab">
                <div class="stats-grid">
                    <!-- Total Visits -->
                    <div class="stat-card glow-blue">
                        <div class="stat-icon">
                            <i class="fas fa-eye"></i>
                        </div>
                        <div class="stat-info">
                            <h3>Total Visits</h3>
                            <div class="stat-number"><?php echo number_format($stats['total_visits']); ?></div>
                            <div class="stat-trend">All Time</div>
                        </div>
                    </div>

                    <!-- Today's Visits -->
                    <div class="stat-card glow-green">
                        <div class="stat-icon">
                            <i class="fas fa-calendar-day"></i>
                        </div>
                        <div class="stat-info">
                            <h3>Today's Visits</h3>
                            <div class="stat-number"><?php echo number_format($stats['today_visits']); ?></div>
                            <div class="stat-trend">24 Hours</div>
                        </div>
                    </div>

                    <!-- Human Traffic -->
                    <div class="stat-card glow-purple">
                        <div class="stat-icon">
                            <i class="fas fa-user"></i>
                        </div>
                        <div class="stat-info">
                            <h3>Human Traffic</h3>
                            <div class="stat-number"><?php echo number_format($classification_stats['human'] ?? 0); ?></div>
                            <div class="stat-percentage">
                                <?php 
                                $total_24h = array_sum($classification_stats);
                                echo $total_24h > 0 ? round(($classification_stats['human'] ?? 0) / $total_24h * 100) : 0;
                                ?>%
                            </div>
                        </div>
                    </div>

                    <!-- Bot Traffic -->
                    <div class="stat-card glow-red">
                        <div class="stat-icon">
                            <i class="fas fa-robot"></i>
                        </div>
                        <div class="stat-info">
                            <h3>Bot Traffic</h3>
                            <div class="stat-number"><?php echo number_format($classification_stats['bot'] ?? 0); ?></div>
                            <div class="stat-percentage">
                                <?php 
                                echo $total_24h > 0 ? round(($classification_stats['bot'] ?? 0) / $total_24h * 100) : 0;
                                ?>%
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Traffic Distribution -->
                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-chart-pie"></i> Traffic Distribution (Last 24h)</h3>
                    </div>
                    <div class="card-body">
                        <div class="traffic-distribution">
                            <?php foreach ($classification_stats as $type => $count): ?>
                            <div class="traffic-item">
                                <div class="traffic-type">
                                    <span class="classification-<?php echo $type; ?>">
                                        <?php echo ucfirst($type); ?>
                                    </span>
                                    <span class="traffic-count"><?php echo number_format($count); ?></span>
                                </div>
                                <div class="traffic-bar">
                                    <div class="bar-fill bar-<?php echo $type; ?>" 
                                         style="width: <?php echo $total_24h > 0 ? ($count / $total_24h * 100) : 0; ?>%">
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <!-- Recent Visits -->
                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-history"></i> Recent Visits</h3>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Time</th>
                                        <th>Device</th>
                                        <th>Country</th>
                                        <th>Classification</th>
                                        <th>Redirect</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_visits as $visit): ?>
                                    <tr>
                                        <td><?php echo date('H:i:s', strtotime($visit['created_at'])); ?></td>
                                        <td><?php echo ucfirst($visit['device_type']); ?></td>
                                        <td><?php echo $visit['country_code'] ?? 'Unknown'; ?></td>
                                        <td class="classification-<?php echo $visit['classification']; ?>">
                                            <?php echo ucfirst($visit['classification']); ?>
                                        </td>
                                        <td><?php echo $visit['link_title'] ?? 'None'; ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Redirect Links Tab -->
            <div class="tab-content" id="links-tab">
                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-plus-circle"></i> Add New Redirect Link</h3>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            <div class="form-grid">
                                <div class="form-group">
                                    <label>Title</label>
                                    <input type="text" name="title" required placeholder="Primary Destination">
                                </div>
                                <div class="form-group">
                                    <label>Destination URL</label>
                                    <input type="url" name="url" required placeholder="https://example.com">
                                </div>
                                <div class="form-group">
                                    <label>Priority (1=highest)</label>
                                    <input type="number" name="priority" value="1" min="1" required>
                                </div>
                                <div class="form-group">
                                    <label>Weight (for distribution)</label>
                                    <input type="number" name="weight" value="100" min="1" required>
                                </div>
                                <div class="form-group">
                                    <label>Status</label>
                                    <select name="status">
                                        <option value="active">Active</option>
                                        <option value="backup">Backup</option>
                                        <option value="emergency">Emergency</option>
                                        <option value="disabled">Disabled</option>
                                    </select>
                                </div>
                            </div>
                            <button type="submit" name="add_link" class="btn-primary">
                                <i class="fas fa-plus"></i> Add Link
                            </button>
                        </form>
                    </div>
                </div>

                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-list"></i> Manage Redirect Links</h3>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Title</th>
                                        <th>URL</th>
                                        <th>Priority</th>
                                        <th>Weight</th>
                                        <th>Status</th>
                                        <th>Redirects</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while ($link = $links->fetch_assoc()): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($link['title']); ?></td>
                                        <td style="max-width: 300px; overflow: hidden; text-overflow: ellipsis;">
                                            <?php echo htmlspecialchars($link['url']); ?>
                                        </td>
                                        <td><?php echo $link['priority']; ?></td>
                                        <td><?php echo $link['weight']; ?></td>
                                        <td>
                                            <span class="status-badge status-<?php echo $link['status']; ?>">
                                                <?php echo ucfirst($link['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo number_format($link['total_redirects']); ?></td>
                                        <td>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                                <input type="hidden" name="link_id" value="<?php echo $link['id']; ?>">
                                                <button type="submit" name="delete_link" class="btn-icon danger" 
                                                        onclick="return confirm('Delete this redirect link?')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Analytics Tab -->
            <div class="tab-content" id="analytics-tab">
                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-chart-network"></i> Link Performance</h3>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Link</th>
                                        <th>Status</th>
                                        <th>Total Redirects</th>
                                        <th>Successful</th>
                                        <th>Success Rate</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($link_stats as $link): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($link['title']); ?></td>
                                        <td>
                                            <span class="status-badge status-<?php echo $link['status']; ?>">
                                                <?php echo ucfirst($link['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo number_format($link['redirects']); ?></td>
                                        <td><?php echo number_format($link['successful_redirects']); ?></td>
                                        <td>
                                            <?php 
                                            $rate = $link['redirects'] > 0 ? ($link['successful_redirects'] / $link['redirects'] * 100) : 0;
                                            echo round($rate, 2) . '%';
                                            ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-globe-americas"></i> Top Countries</h3>
                    </div>
                    <div class="card-body">
                        <div class="countries-grid">
                            <?php foreach ($country_stats as $country): ?>
                            <div class="country-item">
                                <div class="country-flag">
                                    <?php echo strtoupper($country['country_code']); ?>
                                </div>
                                <div class="country-name">
                                    <?php echo $GLOBALS['COUNTRY_CODES'][$country['country_code']] ?? $country['country_code']; ?>
                                </div>
                                <div class="country-count">
                                    <?php echo number_format($country['count']); ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Whitelist Tab -->
            <div class="tab-content" id="whitelist-tab">
                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-plus-circle"></i> Add IP to Whitelist (Testing Mode)</h3>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            <div class="form-grid">
                                <div class="form-group">
                                    <label>IP Address</label>
                                    <input type="text" name="ip_address" required placeholder="192.168.1.1">
                                </div>
                                <div class="form-group">
                                    <label>Note (Optional)</label>
                                    <input type="text" name="note" placeholder="Testing / Development">
                                </div>
                            </div>
                            <button type="submit" name="add_whitelist" class="btn-primary">
                                <i class="fas fa-shield-alt"></i> Add to Whitelist
                            </button>
                        </form>
                    </div>
                </div>

                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-list-check"></i> Whitelisted IPs</h3>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>IP Address</th>
                                        <th>Note</th>
                                        <th>Added By</th>
                                        <th>Date Added</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while ($wl = $whitelist->fetch_assoc()): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($wl['ip_address']); ?></td>
                                        <td><?php echo htmlspecialchars($wl['note']); ?></td>
                                        <td><?php echo htmlspecialchars($wl['added_by']); ?></td>
                                        <td><?php echo date('Y-m-d H:i', strtotime($wl['added_at'])); ?></td>
                                        <td>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                                <input type="hidden" name="whitelist_id" value="<?php echo $wl['id']; ?>">
                                                <button type="submit" name="delete_whitelist" class="btn-icon danger" 
                                                        onclick="return confirm('Remove this IP from whitelist?')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Settings Tab -->
            <div class="tab-content" id="settings-tab">
                <div class="card glow">
                    <div class="card-header">
                        <h3><i class="fas fa-cogs"></i> System Settings</h3>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            
                            <div class="settings-section">
                                <h4><i class="fas fa-redo"></i> Redirect Settings</h4>
                                <div class="form-grid">
                                    <div class="form-group">
                                        <label>System Status</label>
                                        <select name="settings[system_enabled]">
                                            <option value="1" <?php echo $settings['system_enabled'] == '1' ? 'selected' : ''; ?>>Enabled</option>
                                            <option value="0" <?php echo $settings['system_enabled'] == '0' ? 'selected' : ''; ?>>Disabled</option>
                                        </select>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Redirect Mode</label>
                                        <select name="settings[redirect_mode]">
                                            <option value="failover" <?php echo $settings['redirect_mode'] == 'failover' ? 'selected' : ''; ?>>Failover (Primary → Backup → Emergency)</option>
                                            <option value="round-robin" <?php echo $settings['redirect_mode'] == 'round-robin' ? 'selected' : ''; ?>>Round Robin</option>
                                            <option value="random" <?php echo $settings['redirect_mode'] == 'random' ? 'selected' : ''; ?>>Random</option>
                                            <option value="weighted" <?php echo $settings['redirect_mode'] == 'weighted' ? 'selected' : ''; ?>>Weighted Distribution</option>
                                        </select>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Redirect Delay (milliseconds)</label>
                                        <input type="number" name="settings[redirect_delay]" 
                                               value="<?php echo $settings['redirect_delay']; ?>" min="0" max="5000">
                                        <span class="help-text">0 = instant redirect</span>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="checkbox">
                                            <input type="checkbox" name="settings[failover_enabled]" value="1" 
                                                   <?php echo $settings['failover_enabled'] == '1' ? 'checked' : ''; ?>>
                                            <span>Enable Failover System</span>
                                        </label>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="settings-section">
                                <h4><i class="fas fa-robot"></i> Security & Filtering</h4>
                                <div class="form-grid">
                                    <div class="form-group">
                                        <label class="checkbox">
                                            <input type="checkbox" name="settings[bot_protection]" value="1" 
                                                   <?php echo $settings['bot_protection'] == '1' ? 'checked' : ''; ?>>
                                            <span>Enable Advanced Bot Protection</span>
                                        </label>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="checkbox">
                                            <input type="checkbox" name="settings[country_filter]" value="1" 
                                                   <?php echo $settings['country_filter'] == '1' ? 'checked' : ''; ?>>
                                            <span>Enable Country Filtering</span>
                                        </label>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Allowed Countries (comma-separated)</label>
                                        <input type="text" name="settings[allowed_countries]" 
                                               value="<?php echo htmlspecialchars($settings['allowed_countries']); ?>"
                                               placeholder="US,GB,CA">
                                        <span class="help-text">Default: US only</span>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label class="checkbox">
                                            <input type="checkbox" name="settings[privacy_mode]" value="1" 
                                                   <?php echo $settings['privacy_mode'] == '1' ? 'checked' : ''; ?>>
                                            <span>Privacy Mode (Don't store IP addresses)</span>
                                        </label>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="settings-section">
                                <h4><i class="fas fa-user-shield"></i> Admin Security</h4>
                                <div class="form-grid">
                                    <div class="form-group">
                                        <label class="checkbox">
                                            <input type="checkbox" name="settings[admin_ip_restrict]" value="1" 
                                                   <?php echo $settings['admin_ip_restrict'] == '1' ? 'checked' : ''; ?>>
                                            <span>Restrict Admin Access by IP</span>
                                        </label>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Allowed Admin IPs (comma-separated)</label>
                                        <input type="text" name="settings[admin_allowed_ips]" 
                                               value="<?php echo htmlspecialchars($settings['admin_allowed_ips']); ?>"
                                               placeholder="192.168.1.1,203.0.113.5">
                                        <span class="help-text">Your IP: <?php echo $_SERVER['REMOTE_ADDR']; ?></span>
                                    </div>
                                </div>
                            </div>
                            
                            <button type="submit" name="update_settings" class="btn-primary btn-lg">
                                <i class="fas fa-save"></i> Save All Settings
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </main>

        <!-- Footer -->
        <footer class="dashboard-footer">
            <div class="footer-left">
                <span class="status-indicator">
                    <i class="fas fa-circle"></i>
                    System: <?php echo $settings['system_enabled'] == '1' ? 'ACTIVE' : 'DISABLED'; ?>
                </span>
            </div>
            <div class="footer-center">
                <span class="copyright">
                    &copy; <?php echo date('Y'); ?> Shadow PRO v<?php echo SYSTEM_VERSION; ?>
                </span>
            </div>
            <div class="footer-right">
                <span class="server-time">
                    <i class="fas fa-clock"></i>
                    <?php echo date('Y-m-d H:i:s'); ?>
                </span>
            </div>
        </footer>
    </div>

    <!-- JavaScript -->
    <script>
        // Tab Navigation
        document.querySelectorAll('.nav-tab').forEach(tab => {
            tab.addEventListener('click', () => {
                // Remove active class from all tabs
                document.querySelectorAll('.nav-tab').forEach(t => t.classList.remove('active'));
                document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                
                // Add active class to clicked tab
                tab.classList.add('active');
                const tabId = tab.getAttribute('data-tab');
                document.getElementById(`${tabId}-tab`).classList.add('active');
            });
        });
        
        // Real-time Clock
        function updateClock() {
            const now = new Date();
            const timeString = now.toISOString().replace('T', ' ').substr(0, 19);
            document.querySelector('.server-time').innerHTML = 
                `<i class="fas fa-clock"></i> ${timeString}`;
        }
        setInterval(updateClock, 1000);
        
        // Auto-refresh stats on dashboard tab
        let refreshInterval;
        
        function refreshStats() {
            // AJAX call to refresh stats (simplified)
            fetch('?ajax=stats&t=' + Date.now())
                .then(response => response.json())
                .then(data => {
                    // Update stats dynamically
                    const statNumbers = document.querySelectorAll('.stat-number');
                    if (statNumbers[0]) statNumbers[0].textContent = data.total.toLocaleString();
                    if (statNumbers[1]) statNumbers[1].textContent = data.today.toLocaleString();
                    if (statNumbers[2]) statNumbers[2].textContent = data.human.toLocaleString();
                    if (statNumbers[3]) statNumbers[3].textContent = data.bot.toLocaleString();
                })
                .catch(console.error);
        }
        
        // Start auto-refresh on dashboard tab
        function startAutoRefresh() {
            if (refreshInterval) clearInterval(refreshInterval);
            refreshInterval = setInterval(refreshStats, 30000); // 30 seconds
            refreshStats(); // Initial load
        }
        
        // Stop auto-refresh when not on dashboard
        document.querySelectorAll('.nav-tab').forEach(tab => {
            tab.addEventListener('click', () => {
                if (tab.getAttribute('data-tab') === 'dashboard') {
                    startAutoRefresh();
                } else {
                    if (refreshInterval) clearInterval(refreshInterval);
                }
            });
        });
        
        // Initialize if on dashboard
        if (document.querySelector('.nav-tab.active').getAttribute('data-tab') === 'dashboard') {
            startAutoRefresh();
        }
        
        // 3D Card Effects
        document.querySelectorAll('.card').forEach(card => {
            card.addEventListener('mousemove', (e) => {
                const rect = card.getBoundingClientRect();
                const x = e.clientX - rect.left;
                const y = e.clientY - rect.top;
                
                const centerX = rect.width / 2;
                const centerY = rect.height / 2;
                
                const rotateY = ((x - centerX) / centerX) * 3;
                const rotateX = ((centerY - y) / centerY) * 3;
                
                card.style.transform = `
                    perspective(1000px)
                    rotateX(${rotateX}deg)
                    rotateY(${rotateY}deg)
                    translateZ(10px)
                `;
            });
            
            card.addEventListener('mouseleave', () => {
                card.style.transform = 'perspective(1000px) rotateX(0) rotateY(0) translateZ(0)';
            });
        });
        
        // Initialize
        document.addEventListener('DOMContentLoaded', () => {
            updateClock();
            
            // Add background particles
            createBackgroundParticles();
        });
        
        function createBackgroundParticles() {
            const container = document.createElement('div');
            container.className = 'background-particles';
            container.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                pointer-events: none;
                z-index: -1;
            `;
            document.body.appendChild(container);
            
            for (let i = 0; i < 20; i++) {
                const particle = document.createElement('div');
                particle.style.cssText = `
                    position: absolute;
                    width: 2px;
                    height: 2px;
                    background: var(--neon-blue);
                    border-radius: 50%;
                    filter: blur(1px);
                    animation: particleFloat ${5 + Math.random() * 10}s infinite linear;
                `;
                particle.style.left = Math.random() * 100 + 'vw';
                particle.style.top = Math.random() * 100 + 'vh';
                particle.style.animationDelay = Math.random() * 5 + 's';
                container.appendChild(particle);
            }
            
            const style = document.createElement('style');
            style.textContent = `
                @keyframes particleFloat {
                    0% {
                        transform: translateY(0) translateX(0);
                        opacity: 0;
                    }
                    10% {
                        opacity: 1;
                    }
                    90% {
                        opacity: 1;
                    }
                    100% {
                        transform: translateY(-100vh) translateX(${Math.random() * 100 - 50}px);
                        opacity: 0;
                    }
                }
            `;
            document.head.appendChild(style);
        }
    </script>
</body>
</html>
<?php
$db->close();
?>